--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Buildings
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

-- Building class definition
LBuilding =
{
	__tostring = function(t) return "{Building:" .. t.name .. "}" end,
	_ByName = {},
	_ByIndex = {},
}

-------------------------------------------------------------------------------
-- Functions for data description

function GenericBuilding(t) return LBuilding:new(t) end

function EmptyBuilding(t)
	local b = LBuilding:ByName("_empty")
	if b then
		if type(t.character) == "string" then
			-- Empty Buildings have a separate "owner" who takes priority
			t.owner = t.character
			t.character = { t.character }
		elseif type(t.character) == "table" then
			t.owner = t.character[1]
		else
			t.character = {}
		end
		for _,c in ipairs(b.character) do table.insert(t.character,c) end
	end
	return LBuilding:new(t)
end

-------------------------------------------------------------------------------
-- "static" functions to access global Building lists

function LBuilding:ByName(name)
	return self._ByName[name]
end

-------------------------------------------------------------------------------
-- "constructor"

function LBuilding:subclass(t)
	-- Standard object creation...
	t = t or {}
	setmetatable(t, self)
	self.__index = self
	return t
end

function LBuilding:new(t)
	-- Standard object creation...
	t = t or {} setmetatable(t, self) self.__index = self
	
	-- Check parameters
	if not t.name then
		bsgDevWarning("Building defined with no name")
	elseif self._ByName[t.name] then
		bsgDevWarning("Building " .. t.name .. " already defined")
	else
		-- Keep global tables
		self._ByName[t.name] = t
		table.insert(self._ByIndex, t)
		
		-- make sure character is a table
		if type(t.character) == "string" then t.character = { t.character } end

		return t
	end
	
	return nil
end

-------------------------------------------------------------------------------
-- Data post-processing

function LBuilding:PostProcessCharacters(t)
	if t then
		for i,c in ipairs(t) do
			if type(c) == "string" then
				local char = LCharacter:ByName(c)
				if char then t[i] = char end
			end
		end
	end
end

function LBuilding:PostProcessAll()
	for _,b in ipairs(self._ByIndex) do
		-- Convert characters from names to LCharacter references
		self:PostProcessCharacters(b.character)
		self:PostProcessCharacters(b.rank0)
		self:PostProcessCharacters(b.rank1)
		self:PostProcessCharacters(b.rank2)
		self:PostProcessCharacters(b.rank3)
		self:PostProcessCharacters(b.rank4)
		self:PostProcessCharacters(b.rank5)
		
		if b.owner then
			b.owner = LCharacter:ByName(b.owner)
		end
	end
end

function LBuilding:SetPort(p)
	self.port = p
end

-------------------------------------------------------------------------------
-- Popups

function LBuilding:NameRolloverTarget()
	return "(LBuilding:ByName('"..self.name.."')):NameRollover()"
end

function LBuilding:NameRollover()
	StopPulsing()
	return MakeRollover
	{
		x=0,y=0, color=PopupColor, alpha=1, inset=2,
		AppendStyle { font = popupFont, flags=kHAlignLeft+kVAlignTop },
		TightText { x=0,y=0, label=self.name },
	}
end

-------------------------------------------------------------------------------
-- Reset

function LBuilding:ResetOne()
	self.owned = nil
end

function LBuilding:ResetAll()
	for _,b in ipairs(self._ByIndex) do
		b:ResetOne()
	end
end

-------------------------------------------------------------------------------
-- Ownership

function LBuilding:MarkOwned()
	self.owned = true
end

-------------------------------------------------------------------------------
-- Load and Save (structured for class inheritance and override)

function LBuilding:SaveOne()
	local t = nil
	if self.owned then t = { owned=true } end
	return t
end

function LBuilding:BuildSaveTable()
	local t = {}
	for _,b in ipairs(self._ByIndex) do
		local saved = b:SaveOne()
		if saved then t[b.name] = saved end
	end
	return t
end

function LBuilding:LoadOne(t)
	if t and t.owned then self:MarkOwned() end
end

function LBuilding:LoadSaveTable(t)
	for name,data in pairs(t) do
		local b = LBuilding:ByName(name)
		b:LoadOne(data)
	end
end

-------------------------------------------------------------------------------
-- Building appearance

function LBuilding:PrepareForLayout(labelx,labely)
	if self.x and self.y then
		return function() bsgAddSprite { x=self.x,y=self.y,image="ports/"..self.name,
			labelx=labelx,labely=labely,
			contents=self:NameRolloverTarget(),
			command = function() StopPulsing() self:OnActivate() end,
		} end
	else
		return nil
	end
end

-------------------------------------------------------------------------------
-- Building interaction

function LBuilding:SetAmbient()
	if self.ambient then SetAmbient(self.ambient) end
end

function LBuilding:OnActivate()
	gActiveBuilding = self
	self:SetAmbient()
	local char,quest = self:SelectCharacter()
	if not quest then
		-- If rank is 0, use backtotask
		if gSim.rank == 0 and gSim.quest then
			local message = GetString("backtotask")
			DisplayDialog { "ui/chartalk.lua", char=char, fullbody=message, ok="ok" }
		-- If hint on current quest is applicable, give it
		elseif gSim.quest and gSim.quest.hint and gSim.hintWeek <= gSim.weeks and (not char.evil) then
			gSim.hintWeek = gSim.hintWeek + 12
			local message = gSim.quest:HintText()
			DisplayDialog { "ui/chartalk.lua", char=char, fullbody=message, ok="ok" }
		else
			char:DoAction()
		end
	end
	self:ExitBuilding()
end

function LBuilding:ExitBuilding()
	-- Check for quest completion on building exit
	if gSim.quest and self.character then
		for _,endChar in pairs(gSim.quest.ender) do
			for _,bChar in pairs(self.character) do
				if endChar == bChar then
					gSim.quest:CheckCompletion(endChar,false)
					break
				end
			end
		end
	end

	gSim.port:SetAmbient()
	gSim:EvaluateMedals()
	gSim:AutoSave()
	bsgTriggerAd()		-- TOGETHER
	
	gActiveBuilding = nil
	if not gTravelActive then gSim:CheckLoseCondition() end
end

-------------------------------------------------------------------------------
-- Quests

local function NoQuestLimitReached()
	local noQuestWeeks = gSim.weeks - gSim.questWeek
	
	-- Default: 12 weeks max between quests
	-- At rank 1, if going for a while, drive quests quickly to entrepreneur promotion
	-- At rank 3, bring quests more quickly
	local noQuestLimit = 12
	if gSim.rank == 1 and gSim.weeks > 45 then noQuestLimit = 2
	elseif gSim.rank >= 3 then noQuestLimit = 10
	end
	
	if noQuestWeeks >= noQuestLimit then return true
	else return false
	end
end

function LBuilding:ForceEncounter()
	-- Return true if we should force an encounter in this building
	-- Used exclusively (at this point) to override the 20% travel encounter rule

	-- NOTE: ForceQuests do NOT force an encounter, only the quest timer does that...
	
	local available = false
	if NoQuestLimitReached() then
		for _,c in ipairs(self.character) do
			for _,q in ipairs(c.quests) do
				if q:IsEligible() then
				
--DebugOut("FORCE ENCOUNTER:"..tostring(q))				
				
					available = true
					break
				end
			end
			if available then break end
		end
	end
	return available
end

function LBuilding:SelectCharacterFreePlay()
	local char = nil
	
	if self.character then
		-- Make sure at least one character in the building has an action
		local ok = true
		for _,c in ipairs(self.character) do
			if table.getn(c.action) > 0 then
				-- "ok=false" means allow the character check
				if not c.dispatchRank then ok = false break					-- at least one action
				elseif c.dispatchRank > gSim.rank then ok = false break		-- not dispatched yet
				end
			end
		end
		
		while not ok do
			local i = table.getn(self.character)
			i = bsgRandom(1,i)
			char = self.character[i]
			ok = true

			-- Skip characters with no actions
			if (table.getn(char.action) == 0) and (table.getn(self.character) > 1) then ok = false end
		end
	end

	if not char then
		local empty = self._ByName["_empty"]
		if empty then char = empty:SelectCharacterFreePlay() end
	end
	
	-- Return the results
	return char
end

function LBuilding:SelectCharacter()
	local quest = nil
	local char = nil
	
	-- Check for quest-ending character here
	if gSim.quest and self.character then
		for _,endChar in pairs(gSim.quest.ender) do
			for _,bChar in pairs(self.character) do
				if endChar == bChar then
					char = endChar
					break
				end
			end
		end
		
		if char then
			-- Don't show this character if weeks required by quest are not up
			if gSim.requireWeek > gSim.weeks then
				char = nil
			else
				quest = gSim.quest
				gSim.quest:CheckCompletion(char,true)
			end
		end
	end
	
	-- In Free Play mode, don't offer quests under normal conditions
	if not quest and gSim.mode == "free" then
		return self:SelectCharacterFreePlay(),nil
	end
	
	-- Check for forceOffer quest availability and offer anything here
	if not quest and not gSim.quest and self.character then
		-- Determine how long it has been since a quest was completed
		local noQuestWeeks = gSim.weeks - gSim.questWeek

		local chars = {}
		local quests = {}
		
		-- PRIORITY ONE: "Force" quests
		for _,c in ipairs(self.character) do
			for _,q in ipairs(c.quests) do
				if q:CheckForceOffer(noQuestWeeks) and q:IsEligible() then
					table.insert(chars, c)
					table.insert(quests, q)
				end
			end
		end
		
		-- PRIORITY TWO: Building owner quests
		if self.owner and table.getn(quests) == 0 then
			for _,q in ipairs(self.owner.quests) do
				if q:IsEligible() then
					table.insert(quests, q)
					table.insert(chars, self.owner)
				end
			end
		end
		
		-- PRIORITY THREE: Force a quest based on timeout
		if table.getn(quests) == 0 and NoQuestLimitReached() then
			for _,c in ipairs(self.character) do
				for _,q in ipairs(c.quests) do
					if q:IsEligible() then
						table.insert(chars, c)
						table.insert(quests, q)
					end
				end
			end
		end
		
		-- Now, chose from the gathered quests
		local i = table.getn(quests)
		if i > 0 and not quest then
			-- Pick a quest at random
			i = bsgRandom(1,i)
			char = chars[i]
			quest = quests[i]
			quest:Offer(char)
		end
	end

	-- If nothing done, select a character at random, let them offer a quest if available
	if not quest and self.character then
		local ok = false
		char = nil
	
		-- TOTAL HACK: There is a problem here because if more than one character, none of which
		-- have actions, the game will enter an infinite loop below trying to select a character
		-- when the player is on a quest
		if gSim.quest and table.getn(self.character) > 1 then
			ok = true				-- Disable the character check
			for n=1,table.getn(self.character) do
				local c = self.character[n]
				if table.getn(c.action) > 0 then
					-- "ok=false" means allow the character check
					if not c.dispatchRank then ok = false break					-- at least one action
					elseif c.dispatchRank > gSim.rank then ok = false break		-- not dispatched yet
					end
				end
			end
		end
		
		while not ok do
			ok = true
			local i = table.getn(self.character)
			i = bsgRandom(1,i)
			char = self.character[i]
			
			-- And, if that person has quests, offer one if not already on a quest
			if not gSim.quest then
				local quests = {}
				for _,q in ipairs(char.quests) do
					if q:IsEligible() then table.insert(quests, q) end
				end
				i = table.getn(quests)
				if i > 0 then
					i = bsgRandom(1,i)
					quest = quests[i]
					quest:Offer(char)
				end
			end
			
			-- If chosen character ends the current quest, that character has already been ruled out...
			if gSim.quest and gSim.requireWeek > gSim.weeks and char:EndsQuest(gSim.quest) then
--				DebugOut("Skip (requireWeeks quest ender): "..tostring(char))
				ok = false
			end
			
			-- If chosen character has been dispatched, rule them out
			if char.dispatchRank and char.dispatchRank <= gSim.rank then
				ok = false
			end
			
			-- If chosen character does NOT have quests, make sure they at least have an action
			-- This isn't ideal since it relies on random to eventually come up with something different
			if (not quest) and (table.getn(char.action) == 0) and (table.getn(self.character) > 1) then
--				DebugOut("Skip (no action): "..tostring(char))
				ok = false
			end
		end
	end
	
	if not char then
		local empty = self._ByName["_empty"]
		if empty then char,quest = empty:SelectCharacter() end
	end
	
	-- Return the results
	return char,quest
end

